/* ###
 * IP: GHIDRA
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/* Generated by Together */

package ghidra.util.datastruct;
import java.io.Serializable;
import java.util.Arrays;

/**
 * Class to manage multiple linked lists of int indexes. Users can add indexes
 * to a list, remove indexes from a list, remove all indexes from a list, and
 * retrieve all indexes within a given list.
 *
 */
public class IntListIndexer implements Serializable {

    private static final int END_OF_LIST = -1;


    private int []heads;   // array containing indexes that are the head of a list.
    private int []links;   // array of links to other entries
    private int freePtr;   // this is the pointer to the next free entry
    private int size;      // the current number of items in all the lists.
    private int capacity;  // the current size of the array used to store values and links;
    private int numLists;  // the current number of lists that are being managed.


    /**
     * The constructor
     * @param numLists - The initial number of lists to be managed.
     * @param capacity - The current size of the pool of possible indexes.  All indexes
     *  begin on the free list.
     */
    public IntListIndexer(int numLists, int capacity) {

    	this.capacity = capacity;
        this.numLists = numLists;
        links = new int[capacity];
        heads = new int[numLists];
        clear();

    }

    /**
     * Allocates a new index resource and adds it to the front of the linked list
     * indexed by listID.
     * @param listID the id of the list to add to.
     * @exception IndexOutOfBoundsException thrown if the listID is not in the
     * the range [0, numLists).
     */
	public int add(int listID) {
        if ((listID < 0) || (listID >= numLists)) {
            throw new IndexOutOfBoundsException();
        }

		int index=allocate();
        if (index >= 0) {
            links[index] = heads[listID];
		    heads[listID] = index;
        }
		return index;
	}

    /**
     * Allocates a new index resource and adds it to the end of the linked list
     * indexed by listID.
     * @param listID the id of the list to add to.
     * @throws IndexOutOfBoundsException thrown if the listID is not in the
     * the range [0, numLists).
     */
    public int append(int listID) {
        if ((listID < 0) || (listID >= numLists)) {
            throw new IndexOutOfBoundsException();
        }

        int index = allocate();
        if (index >= 0) {
            if (heads[listID] == END_OF_LIST) {
                heads[listID] = index;
            }
            else {
                int p = heads[listID];
                while(links[p] != END_OF_LIST) {
                    p = links[p];
                }
                links[p] = index;
            }
        }
        return index;
    }

   /**
     * Remove the index resource from the linked list indexed by listID.
     * @param listID the id of the list from which to removed the value at index.
     * @param index the index of the value to be removed from the specified list.
     * @exception IndexOutOfBoundsException thrown if the listID is not in the
     * the range [0, numLists).
     */
    public void remove(int listID, int index) {
        if ((listID < 0) || (listID >= numLists)) {
            throw new IndexOutOfBoundsException("The listID is out of bounds");
        }
        if ((index < 0) || (index >= capacity)) {
            throw new IndexOutOfBoundsException();
        }

		int head = heads[listID];

        if (head == END_OF_LIST) {
            return;
        }

        // the special case that the index to be removed is the first one in
        // the list.
        if (head == index) {

            int temp = links[head];
            free(head);
			heads[listID] =temp;
            return;
        }

        int ptr = head;
        // search for the index in the list.  If the end of the list is
        // reached, then the index is not in the list and we don't care.
        while (links[ptr] != END_OF_LIST) {
            if (links[ptr] == index) {
                // found the index to be deleted, remove it from the list by
                // fixing the preivous index's link to skip the removed index.
                links[ptr] = links[index];
                free(index);
				break;
			}
            ptr = links[ptr];
        }

        return;
	}

    /**
     * Removes all indexes from the specified list.
     * @param listID the list to be emptied.
     */
    public void removeAll(int listID) {
        int head = heads[listID];
        heads[listID] = END_OF_LIST;

        // cycle through the list and free all the indexes.
        while (head != END_OF_LIST) {
            int temp = head;
            head = links[head];
            free(temp);
        }
    }


    /**
     * Computes the next size that should be used to grow the index capacity.
     */
    public int getNewCapacity() {
     	int newCapacity;
        if (capacity == Integer.MAX_VALUE) {
            return -1;
        }
        else if (capacity < Integer.MAX_VALUE/2) {

			newCapacity = capacity * 2;
        }
        else{
            newCapacity = Integer.MAX_VALUE;
        }
        return newCapacity;
    }

    /**
     * Returns the current number of used index resources.
     */
    public int getSize() {
        return size;
    }

    /**
     * Returns the current index capacity.
     */
    public int getCapacity() {
        return capacity;
    }

    /**
     * Returns the number of linked list being managed.
     */
    public int getNumLists() {
        return numLists;
    }

    /**
     * Returns the next index resource that follows the given index in a linked list.
     * The index should be an index that is in some linked list.  Otherwise, the
     * results are undefined( probably give you the next index on the free list )
     * @param index the index to search after for the next index.
     * @exception IndexOutOfBoundsException thrown if the index is not in the
     * the range [0, capacity].
     */
    public final int next(int index) {
         return links[index];
    }

    /**
     * Returns the first index resource on the linked list indexed by listID.
     * @exception IndexOutOfBoundsException thrown if the listID is not in the
     * the range [0, numLists].
     */
	public final int first(int listID){
		return heads[listID];
	}

    /**
     * Increases the index resource pool.
     * @param newCapacity the new number of resource indexes to manage.  if this number
     * is smaller than the current number of resource indexes, then nothing changes.
     */
    public void growCapacity(int newCapacity) {

        if (newCapacity <= capacity) {
            return;
        }
        int []temp = new int[newCapacity];
        System.arraycopy(links, 0, temp, 0, capacity);
        for (int i=capacity; i<newCapacity; i++) {
			temp[i] = i + 1;
        }
        temp[newCapacity-1] = END_OF_LIST;
        freePtr=capacity;
        capacity = newCapacity;
        links = temp;
	}

    /**
     * Increases the number of managed linked lists.
     * @param newListSize the new number of linked lists.  If this number is
     * smaller than the current number of linked lists, then nothing changes.
     */
    public void growNumLists(int newListSize) {
        if (newListSize <= numLists) {
            return;
        }
        int[] temp = heads;
        heads = new int[newListSize];
        System.arraycopy(temp, 0, heads, 0, temp.length);
        Arrays.fill(heads,temp.length,heads.length, END_OF_LIST);
        numLists = newListSize;
    }

    /**
     *  Removes all indexes from all lists.
     */
	public void clear() {

		for (int i=0; i<capacity; i++) {
			links[i] = i + 1;
		}
		links[capacity-1] = END_OF_LIST;
		freePtr = 0;
		Arrays.fill(heads, END_OF_LIST);
        size = 0;
	}

    /**
     * Returns the number of indexes in the specified list.
     * @param listID the id of the list from which to get the number of indexes.
     * @exception IndexOutOfBoundsException thrown if the listID is not in the
     * the range [0, numLists).
     */
    public int getListSize(int listID) {

        if ((listID < 0) || (listID >= numLists)) {
            throw new IndexOutOfBoundsException("The listID is out of bounds");
        }

        int count = 0;
        int p = heads[listID];
        while(p != END_OF_LIST) {
            count++;
            p = links[p];
        }
        return count;
    }
    
	/**
     * Returns a free index resource from the free list.  If there are no
     * free index values, then this method will attempt to grow the index
     * resource pool.
     */
    private int allocate(){

    	if (freePtr == END_OF_LIST) {
            growCapacity(getNewCapacity());
            if (freePtr == END_OF_LIST) {
                return END_OF_LIST;
            }
        }
        int p = freePtr;
        freePtr = links[freePtr];
        links[p] = END_OF_LIST;
        ++size;
        return p;
    }

    /**
     * Returns the given index resource to the free list.
     */
    private void free(int p) {
        size--;
        links[p] = freePtr;
        freePtr = p;
    }


}

