/*
 * Decompiled with CFR 0.152.
 */
package htsjdk.samtools.reference;

import htsjdk.samtools.SAMSequenceDictionaryCodec;
import htsjdk.samtools.SAMSequenceRecord;
import htsjdk.samtools.reference.FastaReferenceWriterBuilder;
import htsjdk.samtools.reference.ReferenceSequence;
import htsjdk.samtools.util.SequenceUtil;
import htsjdk.utils.ValidationUtils;
import java.io.IOException;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.Writer;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.nio.file.Path;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.HashSet;
import java.util.Set;
import org.apache.commons.compress.utils.CountingOutputStream;

public final class FastaReferenceWriter
implements AutoCloseable {
    public static final int DEFAULT_BASES_PER_LINE = 60;
    public static final char HEADER_START_CHAR = '>';
    public static final char HEADER_NAME_AND_DESCRIPTION_SEPARATOR = ' ';
    private static final Charset CHARSET = Charset.forName("UTF-8");
    private static final char LINE_SEPARATOR_CHR = '\n';
    private static final char INDEX_FIELD_SEPARATOR_CHR = '\t';
    private static final byte[] LINE_SEPARATOR = String.valueOf('\n').getBytes(CHARSET);
    private final CountingOutputStream fastaStream;
    private final Writer faiIndexWriter;
    private final Writer dictWriter;
    private final MessageDigest md5Digester;
    private final SAMSequenceDictionaryCodec dictCodec;
    private final int defaultBasePerLine;
    private final Set<String> sequenceNames = new HashSet<String>();
    private int currentBasesPerLine;
    private int currentLineBasesCount;
    private long currentBasesCount;
    private long currentSequenceOffset;
    private String currentSequenceName;
    private boolean closed;

    FastaReferenceWriter(int basesPerLine, boolean addMd5, OutputStream fastaOutput, OutputStream indexOutput, OutputStream dictOutput) {
        try {
            this.md5Digester = addMd5 ? MessageDigest.getInstance("MD5") : null;
        }
        catch (NoSuchAlgorithmException e) {
            throw new RuntimeException("Couldn't get md5 algorithm!", e);
        }
        this.defaultBasePerLine = basesPerLine;
        this.fastaStream = new CountingOutputStream(fastaOutput);
        this.faiIndexWriter = indexOutput == null ? NullWriter.NULL_WRITER : new OutputStreamWriter(indexOutput, CHARSET);
        this.dictWriter = dictOutput == null ? NullWriter.NULL_WRITER : new OutputStreamWriter(dictOutput, CHARSET);
        this.dictCodec = new SAMSequenceDictionaryCodec(this.dictWriter);
        this.dictCodec.encodeHeaderLine(false);
    }

    private static void checkSequenceName(String name) {
        ValidationUtils.nonEmpty(name, "Sequence name");
        for (int i = 0; i < name.length(); ++i) {
            char ch = name.charAt(i);
            if (Character.isWhitespace(ch)) {
                throw new IllegalArgumentException("the input name contains blank characters: '" + name + "'");
            }
            if (!Character.isISOControl(ch)) continue;
            throw new IllegalArgumentException("the input name contains control characters: '" + name + "'");
        }
    }

    private static void checkSequenceBases(byte[] bases, int offset, int length) {
        ValidationUtils.nonNull(bases, "input bases");
        ValidationUtils.validateArg(bases.length >= offset + length, "Cannot validate bases beyond end of array.");
        int to = offset + length;
        for (int i = offset; i < to; ++i) {
            byte b = bases[i];
            if (SequenceUtil.isIUPAC(b)) continue;
            throw new IllegalArgumentException("the input sequence contains invalid base calls like: " + (char)b);
        }
    }

    private static String checkDescription(String description) {
        if (description == null || description.isEmpty()) {
            return "";
        }
        for (int i = 0; i < description.length(); ++i) {
            char c = description.charAt(i);
            if (!Character.isISOControl(c) || c == '\t') continue;
            throw new IllegalArgumentException("the input name contains non-tab control characters: '" + description + "'");
        }
        return description;
    }

    public FastaReferenceWriter startSequence(String sequenceName) throws IOException {
        return this.startSequence(sequenceName, "", this.defaultBasePerLine);
    }

    public FastaReferenceWriter startSequence(String sequenceName, int basesPerLine) throws IOException {
        return this.startSequence(sequenceName, "", FastaReferenceWriterBuilder.checkBasesPerLine(basesPerLine));
    }

    public FastaReferenceWriter startSequence(String sequenceName, String description) throws IOException {
        return this.startSequence(sequenceName, description, this.defaultBasePerLine);
    }

    public FastaReferenceWriter startSequence(String sequenceName, String description, int basesPerLine) throws IOException {
        this.assertIsNotClosed();
        FastaReferenceWriter.checkSequenceName(sequenceName);
        String nonNullDescription = FastaReferenceWriter.checkDescription(description);
        FastaReferenceWriterBuilder.checkBasesPerLine(basesPerLine);
        this.closeSequence();
        if (this.sequenceNames.contains(sequenceName)) {
            throw new IllegalStateException("the input sequence name '" + sequenceName + "' has already been added");
        }
        this.currentSequenceName = sequenceName;
        this.currentBasesPerLine = basesPerLine;
        StringBuilder builder = new StringBuilder(sequenceName.length() + nonNullDescription.length() + 2);
        builder.append('>').append(sequenceName);
        if (!nonNullDescription.isEmpty()) {
            builder.append(' ').append(nonNullDescription);
        }
        this.fastaStream.write(builder.toString().getBytes(CHARSET));
        this.fastaStream.write(LINE_SEPARATOR);
        this.currentSequenceOffset = this.fastaStream.getBytesWritten();
        if (this.md5Digester != null) {
            this.md5Digester.reset();
        }
        return this;
    }

    private void closeSequence() throws IOException {
        if (this.currentSequenceName != null) {
            if (this.currentBasesCount == 0L) {
                throw new IllegalStateException("no base was added");
            }
            this.sequenceNames.add(this.currentSequenceName);
            this.writeIndexEntry();
            this.writeDictEntry();
            this.fastaStream.write(LINE_SEPARATOR);
            this.currentBasesCount = 0L;
            this.currentLineBasesCount = 0;
            this.currentSequenceName = null;
        }
    }

    private void writeIndexEntry() throws IOException {
        this.faiIndexWriter.append(this.currentSequenceName).append('\t').append(String.valueOf(this.currentBasesCount)).append('\t').append(String.valueOf(this.currentSequenceOffset)).append('\t').append(String.valueOf(this.currentBasesPerLine)).append('\t').append(String.valueOf(this.currentBasesPerLine + LINE_SEPARATOR.length)).append('\n');
    }

    private void writeDictEntry() {
        SAMSequenceRecord samSequenceRecord = new SAMSequenceRecord(this.currentSequenceName, (int)this.currentBasesCount);
        if (this.md5Digester != null) {
            samSequenceRecord.setMd5(SequenceUtil.md5DigestToString(this.md5Digester.digest()));
        }
        this.dictCodec.encodeSequenceRecord(samSequenceRecord);
    }

    public FastaReferenceWriter appendBases(String basesBases) throws IOException {
        return this.appendBases(basesBases.getBytes(StandardCharsets.US_ASCII));
    }

    public FastaReferenceWriter appendBases(byte[] bases) throws IOException {
        return this.appendBases(bases, 0, bases.length);
    }

    public FastaReferenceWriter appendBases(byte[] bases, int offset, int length) throws IOException {
        int nextLength;
        this.assertIsNotClosed();
        this.assertSequenceOpen();
        FastaReferenceWriter.checkSequenceBases(bases, offset, length);
        ValidationUtils.validateArg(offset >= 0, "the input offset cannot be negative");
        ValidationUtils.validateArg(length >= 0, "the input length must not be negative");
        int to = offset + length;
        ValidationUtils.validateArg(to <= bases.length, "the length + offset goes beyond the end of the input base array: '" + to + "' > '" + bases.length + "'");
        for (int next = offset; next < to; next += nextLength) {
            if (this.currentLineBasesCount == this.currentBasesPerLine) {
                this.fastaStream.write(LINE_SEPARATOR);
                this.currentLineBasesCount = 0;
            }
            nextLength = Math.min(to - next, this.currentBasesPerLine - this.currentLineBasesCount);
            this.fastaStream.write(bases, next, nextLength);
            if (this.md5Digester != null) {
                this.md5Digester.update(new String(bases, next, nextLength).toUpperCase().getBytes());
            }
            this.currentLineBasesCount += nextLength;
        }
        this.currentBasesCount += (long)length;
        return this;
    }

    public FastaReferenceWriter addSequence(ReferenceSequence sequence) throws IOException {
        return this.startSequence(sequence.getName()).appendBases(sequence.getBases());
    }

    public FastaReferenceWriter appendSequence(String name, String description, byte[] bases) throws IOException {
        return this.startSequence(name, description).appendBases(bases);
    }

    public FastaReferenceWriter appendSequence(String name, String description, int basesPerLine, byte[] bases) throws IOException {
        return this.startSequence(name, description, basesPerLine).appendBases(bases);
    }

    private void assertSequenceOpen() {
        if (this.currentSequenceName == null) {
            throw new IllegalStateException("trying to add bases without starting a sequence");
        }
    }

    private void assertIsNotClosed() {
        if (this.closed) {
            throw new IllegalStateException("already closed");
        }
    }

    @Override
    public void close() throws IOException {
        if (!this.closed) {
            try {
                this.closeSequence();
                if (this.sequenceNames.isEmpty()) {
                    throw new IllegalStateException("no sequences were added to the reference");
                }
            }
            finally {
                this.closed = true;
                this.fastaStream.close();
                this.faiIndexWriter.close();
                this.dictWriter.close();
            }
        }
    }

    public static void writeSingleSequenceReference(Path whereTo, boolean makeIndex, boolean makeDict, String name, String description, byte[] bases) throws IOException {
        try (FastaReferenceWriter writer = new FastaReferenceWriterBuilder().setFastaFile(whereTo).setMakeFaiOutput(makeIndex).setMakeDictOutput(makeDict).build();){
            writer.startSequence(name, description);
            writer.appendBases(bases);
        }
    }

    public static void writeSingleSequenceReference(Path whereTo, int basesPerLine, boolean makeIndex, boolean makeDict, String name, String description, byte[] bases) throws IOException {
        try (FastaReferenceWriter writer = new FastaReferenceWriterBuilder().setBasesPerLine(basesPerLine).setFastaFile(whereTo).setMakeFaiOutput(makeIndex).setMakeDictOutput(makeDict).build();){
            writer.startSequence(name, description);
            writer.appendBases(bases);
        }
    }

    private static class NullWriter
    extends Writer {
        public static final NullWriter NULL_WRITER = new NullWriter();

        @Override
        public void write(char[] cbuf, int off, int len) throws IOException {
        }

        @Override
        public void flush() throws IOException {
        }

        @Override
        public void close() throws IOException {
        }

        private NullWriter() {
        }
    }
}

