/*
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2007 Sun Microsystems, Inc. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution, if
 *    any, must include the following acknowlegement:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowlegement may appear in the software itself,
 *    if and wherever such third-party acknowlegements normally appear.
 *
 * 4. The names "The Jakarta Project", "Tomcat", and "Apache Software
 *    Foundation" must not be used to endorse or promote products derived
 *    from this software without prior written permission. For written
 *    permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache"
 *    nor may "Apache" appear in their names without prior written
 *    permission of the Apache Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 *
 * The original contributor of this 
 *
 */

package org.netbeans.microedition.databinding.svg;

import javax.microedition.m2g.SVGImage;
import org.netbeans.microedition.databinding.DataBindingProvider;
import org.netbeans.microedition.svg.SVGPlayer;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.svg.SVGElement;
import org.w3c.dom.svg.SVGMatrix;
import org.w3c.dom.svg.SVGPath;
import org.w3c.dom.svg.SVGRGBColor;
import org.w3c.dom.svg.SVGRect;
import org.w3c.dom.svg.SVGSVGElement;

/**
 *  This class provides binding between SVG graphics and data binding core mechanism.
 *
 *
 * @author breh
 */
public class SVGBindingProvider implements DataBindingProvider {
  
    /**
     * SVGImage feature identifies an element and trait which can be
     * bound to data. 
     *
     */
    public static class SVGImageFeature {
        private String elementID;
        private String trait;
        
        public SVGImageFeature(String elementID, String trait) {
            if (elementID == null) throw new IllegalArgumentException("elementID cannot be null");
            if (trait == null) throw new IllegalArgumentException("trait cannot be null");
            this.elementID = elementID;
            this.trait = trait;
        }
    }
    
    /**
     * Creates a new instance of SVGBindingProvider
     */
    public SVGBindingProvider() {
    }

    public void notifyBind(String elPointer, Object consumer, Object feature) {
    }

    public void notifyUnbind(String elPointer, Object consumer, Object feature) {
    }    
    

    public Object pullDataFromConsumer(String elPointer, Object consumer, Object feature) {
        return null;
    }

    public void pushDataToConsumer(String elPointer, Object consumer, Object feature, Object value) {
        if (consumer instanceof SVGPlayer) {
            SVGPlayer player = (SVGPlayer)consumer;
            if (feature instanceof SVGImageFeature) {
                SVGImageFeature svgImageFeature = (SVGImageFeature)feature;
                safelyUpdateDOM(player, svgImageFeature, value);
            }
        }
    }
    
    
    private void safelyUpdateDOM(SVGPlayer player, final SVGImageFeature feature, final Object value) {
        int state = player.getAnimatorState();
        final SVGImage svgImage = player.getSVGImage();
        final Document doc = svgImage.getDocument();
        final SVGSVGElement svgsvg = (SVGSVGElement)doc.getDocumentElement();
        Element element = doc.getElementById(feature.elementID);
        if (element instanceof SVGElement) {
            final SVGElement svgElement = (SVGElement)element;
            // apply the change
            Runnable runnable = new Runnable() {
                public void run() {
                    setTypeSafeTrait(svgsvg, svgElement, feature.trait, value);
                }            
            };
            player.invokeLaterSafely(runnable);
        }
    }

    
    private static void setTypeSafeTrait(SVGSVGElement svgsvg, SVGElement svgElement, String traitName, Object traitValue) {
        // this is hack, but currently there is no other way how to do this :-(
        if ("fill".equals(traitName)) {
            if (traitValue instanceof Integer) {
               int color = ((Integer)traitValue).intValue();
               traitValue = getRGBColor(svgsvg,color);
            }
        }
        if (traitValue instanceof Float) {
            svgElement.setFloatTrait(traitName,((Float)traitValue).floatValue());
        } else if (traitValue instanceof SVGMatrix) {
            svgElement.setMatrixTrait(traitName,(SVGMatrix)traitValue);
        } else if (traitValue instanceof SVGPath) {
            svgElement.setPathTrait(traitName,(SVGPath)traitValue);
        } else if (traitValue instanceof SVGRGBColor) {            
            svgElement.setRGBColorTrait(traitName,(SVGRGBColor)traitValue);
        } else if (traitValue instanceof SVGRect) {
            svgElement.setRectTrait(traitName,(SVGRect)traitValue);
        } else {
            svgElement.setTrait(traitName,""+traitValue);
        }
    }
    
    
    private static SVGRGBColor getRGBColor(SVGSVGElement svgsvg, int color) {
        int r = (color / 0x10000) & 0xFF;
        int g = (color / 0x100) & 0xFF;
        int b = color & 0xFF;
        return svgsvg.createSVGRGBColor(r,g,b);
    }



}
