/*
 * Decompiled with CFR 0.152.
 */
package com.limegroup.gnutella.tigertree;

import com.google.inject.Inject;
import com.google.inject.Singleton;
import com.limegroup.gnutella.DownloadManager;
import com.limegroup.gnutella.URN;
import com.limegroup.gnutella.http.HTTPHeaderValue;
import com.limegroup.gnutella.library.FileDesc;
import com.limegroup.gnutella.library.IncompleteFileDesc;
import com.limegroup.gnutella.library.Library;
import com.limegroup.gnutella.tigertree.HashTree;
import com.limegroup.gnutella.tigertree.HashTreeCache;
import com.limegroup.gnutella.tigertree.HashTreeFactory;
import java.io.File;
import java.io.IOException;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.Callable;
import java.util.concurrent.CancellationException;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.limewire.collection.Tuple;
import org.limewire.concurrent.ExecutorsHelper;
import org.limewire.concurrent.SimpleFuture;
import org.limewire.util.CommonUtils;
import org.limewire.util.FileUtils;
import org.limewire.util.GenericsUtils;

@Singleton
public final class HashTreeCacheImpl
implements HashTreeCache {
    private static final Log LOG = LogFactory.getLog(HashTreeCacheImpl.class);
    private final ExecutorService QUEUE = ExecutorsHelper.newProcessingQueue("TreeHashTread");
    private final Map<URN, Future<URN>> SHA1_TO_ROOT_MAP = new HashMap<URN, Future<URN>>();
    private final Map<URN, Future<HashTree>> TTREE_MAP = new HashMap<URN, Future<HashTree>>();
    private final File ROOTS_FILE = new File(CommonUtils.getUserSettingsDir(), "ttroot.cache");
    private final File DATA_FILE = new File(CommonUtils.getUserSettingsDir(), "ttdata.cache");
    private volatile boolean dirty = false;
    private final HashTreeFactory tigerTreeFactory;
    private final Library managedFileList;

    @Inject
    HashTreeCacheImpl(HashTreeFactory tigerTreeFactory, Library managedFileList) {
        this.tigerTreeFactory = tigerTreeFactory;
        this.managedFileList = managedFileList;
        Tuple<Map<URN, URN>, Map<URN, HashTree>> tuple = this.loadCaches();
        for (Map.Entry<URN, URN> entry : tuple.getFirst().entrySet()) {
            this.SHA1_TO_ROOT_MAP.put(entry.getKey(), new SimpleFuture<URN>(entry.getValue()));
        }
        for (Map.Entry<URN, HTTPHeaderValue> entry : tuple.getSecond().entrySet()) {
            this.TTREE_MAP.put(entry.getKey(), new SimpleFuture<HTTPHeaderValue>(entry.getValue()));
        }
    }

    public HashTree getHashTreeAndWait(FileDesc fd, long timeout) throws InterruptedException, TimeoutException, ExecutionException {
        if (fd instanceof IncompleteFileDesc) {
            throw new IllegalArgumentException("fd must not inherit from IncompleFileDesc");
        }
        if (fd.getSHA1Urn() != null) {
            Future<HashTree> futureTree = this.getOrScheduleHashTreeFuture(fd);
            return futureTree.get(timeout, TimeUnit.MILLISECONDS);
        }
        return null;
    }

    @Override
    public synchronized HashTree getHashTree(FileDesc fd) {
        if (fd.getSHA1Urn() != null) {
            Future<HashTree> futureTree = this.getOrScheduleHashTreeFuture(fd);
            return this.getTreeFromFuture(fd.getSHA1Urn(), futureTree);
        }
        return null;
    }

    private HashTree getTreeFromFuture(URN sha1, Future<HashTree> futureTree) {
        if (futureTree.isDone()) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("Future tree exists for: " + sha1 + " and is finished");
            }
            try {
                return futureTree.get();
            }
            catch (InterruptedException e) {
                LOG.debug("interrupted while hashing tree", e);
                this.TTREE_MAP.remove(sha1);
            }
            catch (ExecutionException e) {
                LOG.debug("error while hashing tree", e);
                this.TTREE_MAP.remove(sha1);
            }
            catch (CancellationException e) {
                LOG.debug("cancelled while hashing tree", e);
                this.TTREE_MAP.remove(sha1);
            }
            return null;
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("Future tree exists for: " + sha1 + " but is not finished");
        }
        return null;
    }

    private URN getRootFromFuture(URN sha1, Future<URN> futureRoot) {
        if (futureRoot.isDone()) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("Future root exists for: " + sha1 + " and is finished");
            }
            try {
                return futureRoot.get();
            }
            catch (InterruptedException e) {
                LOG.debug("interrupted while hashing root", e);
                this.SHA1_TO_ROOT_MAP.remove(sha1);
            }
            catch (ExecutionException e) {
                LOG.debug("error while hashing root", e);
                this.SHA1_TO_ROOT_MAP.remove(sha1);
            }
            catch (CancellationException e) {
                LOG.debug("cancelled while hashing root", e);
                this.SHA1_TO_ROOT_MAP.remove(sha1);
            }
            return null;
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("Future root exists for: " + sha1 + " but is not finished");
        }
        return null;
    }

    @Override
    public synchronized URN getOrScheduleHashTreeRoot(FileDesc fd) {
        URN sha1 = fd.getSHA1Urn();
        if (sha1 != null) {
            URN root;
            Future<HashTree> futureTree = this.TTREE_MAP.get(sha1);
            Future<URN> futureRoot = this.SHA1_TO_ROOT_MAP.get(sha1);
            HashTree tree = futureTree == null ? null : this.getTreeFromFuture(sha1, futureTree);
            URN uRN = root = futureRoot == null ? null : this.getRootFromFuture(sha1, futureRoot);
            if (tree != null) {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("Returning root from tree");
                }
                return tree.getTreeRootUrn();
            }
            if (root != null) {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("Returning root from future");
                }
                return root;
            }
            if (!(fd instanceof IncompleteFileDesc)) {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("Scheduling: " + sha1 + " for tree root");
                }
                futureRoot = this.QUEUE.submit(new RootRunner(fd));
                this.SHA1_TO_ROOT_MAP.put(sha1, futureRoot);
            } else if (LOG.isDebugEnabled()) {
                LOG.debug("Ignoring: " + sha1 + " because FD is incomplete.");
            }
            return null;
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("Returning null for root because no sha1 for fd: " + fd);
        }
        return null;
    }

    private synchronized Future<HashTree> getOrScheduleHashTreeFuture(FileDesc fd) {
        URN sha1 = fd.getSHA1Urn();
        Future<HashTree> futureTree = this.TTREE_MAP.get(sha1);
        if (futureTree == null && !(fd instanceof IncompleteFileDesc)) {
            Future<URN> futureRoot;
            if (LOG.isDebugEnabled()) {
                LOG.debug("Scheduling: " + sha1 + " for full tree");
            }
            if ((futureRoot = this.SHA1_TO_ROOT_MAP.get(sha1)) != null && !futureRoot.isDone()) {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("Cancelling: " + sha1 + " from root schedule");
                }
                futureRoot.cancel(true);
            }
            futureTree = this.QUEUE.submit(new HashTreeRunner(fd));
            this.TTREE_MAP.put(sha1, futureTree);
        }
        return futureTree;
    }

    @Override
    public synchronized HashTree getHashTree(URN sha1) {
        if (!sha1.isSHA1()) {
            throw new IllegalArgumentException();
        }
        Future<HashTree> futureTree = this.TTREE_MAP.get(sha1);
        if (futureTree != null) {
            return this.getTreeFromFuture(sha1, futureTree);
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("No future tree exists for: " + sha1);
        }
        return null;
    }

    @Override
    public synchronized URN getHashTreeRootForSha1(URN sha1) {
        Future<URN> urnFuture;
        if (!sha1.isSHA1()) {
            throw new IllegalArgumentException();
        }
        HashTree tree = this.getHashTree(sha1);
        if (tree != null) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("Retrieving root from tree for: " + sha1);
            }
            return tree.getTreeRootUrn();
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("Retrieving root from root map for: " + sha1);
        }
        if ((urnFuture = this.SHA1_TO_ROOT_MAP.get(sha1)) != null) {
            return this.getRootFromFuture(sha1, urnFuture);
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("No future root exists for: " + sha1);
        }
        return null;
    }

    @Override
    public synchronized void purgeTree(URN sha1) {
        if (!sha1.isSHA1()) {
            throw new IllegalArgumentException();
        }
        Future<HashTree> futureTree = this.TTREE_MAP.remove(sha1);
        if (futureTree != null) {
            futureTree.cancel(true);
            this.dirty = true;
        }
    }

    @Override
    public synchronized HashTree addHashTree(URN sha1, HashTree tree) {
        boolean shouldAdd = this.hashTreeCalculated(sha1, tree);
        if (shouldAdd) {
            Future oldFuture = this.TTREE_MAP.put(sha1, new SimpleFuture<HashTree>(tree));
            if (oldFuture != null) {
                oldFuture.cancel(true);
            }
            return tree;
        }
        return null;
    }

    private synchronized boolean hashTreeCalculated(URN sha1, HashTree tree) {
        URN root = tree.getTreeRootUrn();
        this.addRoot(sha1, root);
        if (tree.isGoodDepth()) {
            Future<URN> futureRoot = this.SHA1_TO_ROOT_MAP.remove(sha1);
            if (futureRoot != null) {
                futureRoot.cancel(true);
            }
            this.dirty = true;
            if (LOG.isDebugEnabled()) {
                LOG.debug("added hashtree for urn " + sha1 + ";" + tree.getRootHash());
            }
            return true;
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("hashtree for urn " + sha1 + " had bad depth");
        }
        return false;
    }

    @Override
    public synchronized void addRoot(URN sha1, URN ttroot) {
        if (!sha1.isSHA1() || !ttroot.isTTRoot()) {
            throw new IllegalArgumentException();
        }
        Future oldFuture = this.SHA1_TO_ROOT_MAP.put(sha1, new SimpleFuture<URN>(ttroot));
        if (oldFuture != null) {
            oldFuture.cancel(true);
        }
        this.dirty = true;
    }

    private Tuple<Map<URN, URN>, Map<URN, HashTree>> loadCaches() {
        Object trees;
        HashMap roots;
        try {
            roots = this.ROOTS_FILE.exists() ? FileUtils.readObject(this.ROOTS_FILE) : new HashMap();
            trees = this.DATA_FILE.exists() ? FileUtils.readObject(this.DATA_FILE) : new HashMap();
        }
        catch (Throwable t) {
            LOG.debug("Error reading from disk.", t);
            roots = new HashMap();
            trees = new HashMap();
        }
        Map<URN, URN> rootsMap = GenericsUtils.scanForMap(roots, URN.class, URN.class, GenericsUtils.ScanMode.REMOVE);
        Map<URN, HashTree> treesMap = GenericsUtils.scanForMap(trees, URN.class, HashTree.class, GenericsUtils.ScanMode.REMOVE);
        rootsMap.keySet().removeAll(treesMap.keySet());
        Iterator<Object> iter = rootsMap.entrySet().iterator();
        while (iter.hasNext()) {
            Map.Entry<URN, URN> e = iter.next();
            if (e.getKey().isSHA1() && e.getValue().isTTRoot()) continue;
            iter.remove();
        }
        iter = treesMap.keySet().iterator();
        while (iter.hasNext()) {
            URN urn = (URN)iter.next();
            if (urn.isSHA1()) continue;
            iter.remove();
        }
        return new Tuple<Map<URN, URN>, Map<URN, HashTree>>(rootsMap, treesMap);
    }

    private Set<URN> removeOldEntries(Map<URN, URN> roots, Map<URN, HashTree> map, Library library, DownloadManager downloadManager) {
        HashSet<URN> removed = new HashSet<URN>();
        Iterator<URN> iter = roots.keySet().iterator();
        while (iter.hasNext()) {
            URN sha1 = iter.next();
            if (!library.getFileDescsMatching(sha1).isEmpty() || downloadManager.getIncompleteFileManager().getFileForUrn(sha1) != null || Math.random() > (double)(map.size() / 200)) continue;
            removed.add(sha1);
            iter.remove();
            map.remove(sha1);
            this.dirty = true;
        }
        return removed;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void persistCache(Library library, DownloadManager downloadManager) {
        HashMap<URN, URN> roots;
        HashMap<URN, HashTree> trees;
        if (!this.dirty) {
            return;
        }
        HashTreeCacheImpl hashTreeCacheImpl = this;
        synchronized (hashTreeCacheImpl) {
            trees = new HashMap<URN, HashTree>(this.TTREE_MAP.size());
            for (Map.Entry<URN, Future<HashTree>> entry : this.TTREE_MAP.entrySet()) {
                if (!entry.getValue().isDone()) continue;
                try {
                    trees.put(entry.getKey(), entry.getValue().get());
                }
                catch (InterruptedException e) {
                }
                catch (ExecutionException e) {
                }
                catch (CancellationException e) {}
            }
            roots = new HashMap<URN, URN>(this.SHA1_TO_ROOT_MAP.size());
            for (Map.Entry<URN, Future<HTTPHeaderValue>> entry : this.SHA1_TO_ROOT_MAP.entrySet()) {
                if (!entry.getValue().isDone()) continue;
                try {
                    roots.put(entry.getKey(), (URN)entry.getValue().get());
                }
                catch (InterruptedException e) {
                }
                catch (ExecutionException e) {
                }
                catch (CancellationException e) {}
            }
        }
        Set<URN> removed = this.removeOldEntries(roots, trees, library, downloadManager);
        if (!removed.isEmpty()) {
            HashTreeCacheImpl i$ = this;
            synchronized (i$) {
                this.SHA1_TO_ROOT_MAP.keySet().removeAll(removed);
                this.TTREE_MAP.keySet().removeAll(removed);
            }
        }
        try {
            FileUtils.writeObject(this.ROOTS_FILE, roots);
            FileUtils.writeObject(this.DATA_FILE, trees);
            this.dirty = false;
        }
        catch (IOException e) {
            // empty catch block
        }
    }

    private class RootRunner
    implements Callable<URN> {
        private final FileDesc FD;

        RootRunner(FileDesc fd) {
            this.FD = fd;
        }

        @Override
        public URN call() throws IOException, InterruptedException {
            if (HashTreeCacheImpl.this.managedFileList.getFileDescsMatching(this.FD.getSHA1Urn()).isEmpty()) {
                throw new IOException("no FDs with SHA1 anymore.");
            }
            URN ttRoot = URN.createTTRootFile(this.FD.getFile());
            List<FileDesc> fds = HashTreeCacheImpl.this.managedFileList.getFileDescsMatching(this.FD.getSHA1Urn());
            for (FileDesc fd : fds) {
                fd.addUrn(ttRoot);
            }
            HashTreeCacheImpl.this.dirty = true;
            return ttRoot;
        }
    }

    private class HashTreeRunner
    implements Callable<HashTree> {
        private final FileDesc FD;

        HashTreeRunner(FileDesc fd) {
            this.FD = fd;
        }

        @Override
        public HashTree call() throws IOException {
            URN sha1 = this.FD.getSHA1Urn();
            HashTree tree = HashTreeCacheImpl.this.tigerTreeFactory.createHashTree(this.FD);
            HashTreeCacheImpl.this.hashTreeCalculated(sha1, tree);
            return tree;
        }
    }
}

