/* ###
 * IP: GHIDRA
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/// \file inject_ghidra.hh
/// \brief P-code injection classes using a Ghidra client as the back-end for generating p-code

#ifndef __INJECT_GHIDRA__
#define __INJECT_GHIDRA__

#include "pcodeinject.hh"
#include "ghidra_arch.hh"

/// \brief An injection context that can be serialized and sent to the Ghidra client
///
/// This adds the capability to wrap up the context data in \<context> XML tag
/// that can then be forwarded to the Ghidra client.
class InjectContextGhidra : public InjectContext {
public:
  virtual void saveXml(ostream &s) const;
};

/// \brief An injection payload that uses a Ghidra client to generate the p-code ops
///
/// This acts as a placeholder for the actual details of the payload.
/// When the inject() method is invoked, the context is wrapped as XML and
/// sent to the Ghidra client, which returns the actual p-code to inject.
class InjectPayloadGhidra : public InjectPayload {
  string source;		///< Source description to associate with the payload
public:
  InjectPayloadGhidra(const string &src,const string &nm,int4 tp) : InjectPayload(nm,tp) { source = src; }	///< Constructor
  virtual void inject(InjectContext &context,PcodeEmit &emit) const;
  virtual void restoreXml(const Element *el) {}
  virtual void printTemplate(ostream &s) const;
  virtual string getSource(void) const { return source; }
};

/// \brief A call-fixup injection that uses a Ghidra client to generate the p-code ops
class InjectCallfixupGhidra : public InjectPayloadGhidra {
public:
  InjectCallfixupGhidra(const string &src,const string &nm);	///< Constructor
  virtual void restoreXml(const Element *el);
};

/// \brief A callother-fixup injection that uses a Ghidra client to generate the p-code ops
class InjectCallotherGhidra : public InjectPayloadGhidra {
public:
  InjectCallotherGhidra(const string &src,const string &nm);	///< Constructor
  virtual void restoreXml(const Element *el);
};

/// \brief A \e p-code \e script that uses a Ghidra client to generate the p-code ops
///
/// The \<body> section of any p-code script encountered in .cspec or .pspec files
/// is ignored. Instead the emulator is initialized by fetching compiled p-code
/// from the Ghidra client.
class ExecutablePcodeGhidra : public ExecutablePcode {
public:
  ExecutablePcodeGhidra(Architecture *g,const string &src,const string &nm);	///< Constructor
  virtual void inject(InjectContext &context,PcodeEmit &emit) const;
  virtual void restoreXml(const Element *el);
  virtual void printTemplate(ostream &s) const;
};

/// \brief A p-code injection library that uses a Ghidra client to generate/compile the injection p-code
///
/// The InjectPayload objects produced by this library are just placeholders (see InjectPayloadGhidra).
/// At the time of injection, final p-code is generated by the Ghidra client.
class PcodeInjectLibraryGhidra : public PcodeInjectLibrary {
  InjectContextGhidra contextCache;		///< A context object that wraps data in XML for the Ghidra client
  vector<OpBehavior *> inst;			///< Collected behaviors for the ExecutablePcode payloads
  virtual int4 allocateInject(const string &sourceName,const string &name,int4 type);
  virtual void registerInject(int4 injectid);
public:
  PcodeInjectLibraryGhidra(ArchitectureGhidra *ghi);		///< Constructor
  virtual int4 manualCallFixup(const string &name,const string &snippet);
  virtual int4 manualCallOtherFixup(const string &name,const string &outname,const vector<string> &inname,
				    const string &snippet);
  virtual InjectContext &getCachedContext(void) { return contextCache; }
  virtual const vector<OpBehavior *> &getBehaviors(void);
};

#endif
