/*
 * Decompiled with CFR 0.152.
 */
package ch.systemsx.cisd.hdf5.h5ar;

import ch.systemsx.cisd.base.exceptions.IErrorStrategy;
import ch.systemsx.cisd.base.exceptions.IOExceptionUnchecked;
import ch.systemsx.cisd.base.io.IOutputStream;
import ch.systemsx.cisd.base.unix.FileLinkType;
import ch.systemsx.cisd.hdf5.HDF5GenericStorageFeatures;
import ch.systemsx.cisd.hdf5.HDF5OpaqueType;
import ch.systemsx.cisd.hdf5.IHDF5Writer;
import ch.systemsx.cisd.hdf5.IHDF5WriterConfigurator;
import ch.systemsx.cisd.hdf5.h5ar.ArchiveEntry;
import ch.systemsx.cisd.hdf5.h5ar.ArchivingException;
import ch.systemsx.cisd.hdf5.h5ar.ArchivingStrategy;
import ch.systemsx.cisd.hdf5.h5ar.DirectoryIndex;
import ch.systemsx.cisd.hdf5.h5ar.DirectoryIndexUpdater;
import ch.systemsx.cisd.hdf5.h5ar.IArchiveEntryVisitor;
import ch.systemsx.cisd.hdf5.h5ar.IDirectoryIndex;
import ch.systemsx.cisd.hdf5.h5ar.IDirectoryIndexProvider;
import ch.systemsx.cisd.hdf5.h5ar.IdCache;
import ch.systemsx.cisd.hdf5.h5ar.LinkRecord;
import ch.systemsx.cisd.hdf5.h5ar.Utils;
import ch.systemsx.cisd.hdf5.io.HDF5IOAdapterFactory;
import java.io.File;
import java.io.FileInputStream;
import java.io.Flushable;
import java.io.IOException;
import java.io.InputStream;
import java.util.Iterator;
import java.util.List;
import java.util.zip.CRC32;
import ncsa.hdf.hdf5lib.exceptions.HDF5Exception;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.FilenameUtils;
import org.apache.commons.io.IOUtils;

class HDF5ArchiveUpdater {
    private static final String OPAQUE_TAG_FILE = "FILE";
    private static final int SIZEHINT_FACTOR = 5;
    private static final int MIN_GROUP_MEMBER_COUNT_TO_COMPUTE_SIZEHINT = 100;
    private static final int SMALL_DATASET_LIMIT = 4096;
    private final IHDF5Writer hdf5Writer;
    private final IDirectoryIndexProvider indexProvider;
    private final IErrorStrategy errorStrategy;
    private final DirectoryIndexUpdater indexUpdater;
    private final IdCache idCache;
    private final byte[] buffer;

    public HDF5ArchiveUpdater(IHDF5Writer hdf5Writer, IDirectoryIndexProvider indexProvider, IdCache idCache, byte[] buffer) {
        this.hdf5Writer = hdf5Writer;
        this.indexProvider = indexProvider;
        this.idCache = idCache;
        this.errorStrategy = indexProvider.getErrorStrategy();
        this.indexUpdater = new DirectoryIndexUpdater(indexProvider);
        this.buffer = buffer;
    }

    public HDF5ArchiveUpdater archive(File path, ArchivingStrategy strategy, int chunkSize, boolean keepNameFromPath, IArchiveEntryVisitor entryVisitorOrNull) {
        File absolutePath = Utils.normalizePath(path);
        return this.archive(keepNameFromPath ? absolutePath.getParentFile() : absolutePath, absolutePath, strategy, chunkSize, entryVisitorOrNull);
    }

    public IOutputStream archiveFile(String directory, LinkRecord link, boolean compress, int chunkSize) {
        if (link.getLinkType() != FileLinkType.REGULAR_FILE) {
            this.errorStrategy.dealWithError(new ArchivingException("A regular file is expected here."));
        }
        return new H5ARIOutputStream(Utils.normalizePath(directory), link, chunkSize, compress);
    }

    public HDF5ArchiveUpdater archive(String directory, LinkRecord link, InputStream inputOrNull, boolean compress, int chunkSize) {
        boolean ok = true;
        String normalizedDir = Utils.normalizePath(directory);
        String hdf5ObjectPath = Utils.concatLink(normalizedDir, link.getLinkName());
        ArchiveEntry entry = new ArchiveEntry(normalizedDir, hdf5ObjectPath, link, this.idCache);
        boolean groupExists = this.hdf5Writer.isGroup(normalizedDir);
        if (link.getLinkType() == FileLinkType.DIRECTORY) {
            if (inputOrNull == null) {
                ok = this.archiveEmptyDirectory(normalizedDir, link);
            } else {
                this.errorStrategy.dealWithError(new ArchivingException("Cannot take InputStream when archiving a directory."));
            }
        } else if (link.getLinkType() == FileLinkType.SYMLINK) {
            if (inputOrNull == null) {
                ok = this.archiveSymLink(entry);
            } else {
                this.errorStrategy.dealWithError(new ArchivingException("Cannot take InputStream when archiving a symlink."));
            }
        } else if (link.getLinkType() == FileLinkType.REGULAR_FILE) {
            if (inputOrNull != null) {
                HDF5GenericStorageFeatures compression = compress ? HDF5GenericStorageFeatures.GENERIC_DEFLATE : HDF5GenericStorageFeatures.GENERIC_NO_COMPRESSION;
                try {
                    DataSetInfo info = this.copyToHDF5(inputOrNull, hdf5ObjectPath, compression, chunkSize);
                    link.setCrc32(info.crc32);
                    link.setSize(info.size);
                }
                catch (IOException ex) {
                    ok = false;
                    this.errorStrategy.dealWithError(new ArchivingException(hdf5ObjectPath, ex));
                }
                catch (HDF5Exception ex) {
                    ok = false;
                    this.errorStrategy.dealWithError(new ArchivingException(hdf5ObjectPath, ex));
                }
            } else {
                this.errorStrategy.dealWithError(new ArchivingException("Need to have InputStream when archiving a regular file."));
            }
        } else {
            this.errorStrategy.dealWithError(new ArchivingException("Don't know how to archive file link type " + link.getLinkType()));
            ok = false;
        }
        if (ok) {
            this.updateIndicesOnThePath(hdf5ObjectPath, link, groupExists);
        }
        return this;
    }

    public HDF5ArchiveUpdater archive(String rootDirInArchive, File path, ArchivingStrategy strategy, int chunkSize, IArchiveEntryVisitor entryVisitorOrNull) {
        boolean ok;
        File absolutePath = Utils.normalizePath(path);
        String normalizedRootDirInArchive = Utils.normalizePath(rootDirInArchive);
        String hdf5ObjectPath = Utils.concatLink(normalizedRootDirInArchive, absolutePath.getName());
        String hdf5GroupPath = Utils.getParentPath(hdf5ObjectPath);
        boolean groupExists = this.hdf5Writer.isGroup(hdf5GroupPath);
        int crc32 = 0;
        LinkRecord linkOrNull = LinkRecord.tryCreate(absolutePath, this.errorStrategy);
        if (linkOrNull == null) {
            return this;
        }
        ArchiveEntry entry = new ArchiveEntry(normalizedRootDirInArchive, hdf5ObjectPath, linkOrNull, this.idCache);
        if (linkOrNull.isSymLink()) {
            ok = this.archiveSymLink(entry, absolutePath, entryVisitorOrNull);
        } else if (absolutePath.isDirectory()) {
            ok = this.archiveDirectory(absolutePath, entry, strategy, chunkSize, entryVisitorOrNull);
        } else if (absolutePath.isFile()) {
            DataSetInfo dataSetInfoOrNull = this.tryArchiveFile(absolutePath, entry, strategy.getStorageFeatureForPath(hdf5ObjectPath), chunkSize, entryVisitorOrNull);
            boolean bl = ok = dataSetInfoOrNull != null;
            if (dataSetInfoOrNull != null) {
                crc32 = dataSetInfoOrNull.crc32;
            }
        } else {
            ok = false;
            this.errorStrategy.dealWithError(new ArchivingException(absolutePath, new IOException("Path corresponds to neither a file nor a directory.")));
        }
        if (ok) {
            this.indexUpdater.updateIndicesOnThePath(normalizedRootDirInArchive, absolutePath, crc32, groupExists);
        }
        return this;
    }

    public HDF5ArchiveUpdater archiveBelow(String rootDirInArchive, File directory, ArchivingStrategy strategy, int chunkSize, IArchiveEntryVisitor entryVisitorOrNull) {
        File absoluteDirectory = Utils.normalizePath(directory);
        if (absoluteDirectory.isDirectory()) {
            LinkRecord linkOrNull = LinkRecord.tryCreate(absoluteDirectory, this.errorStrategy);
            if (linkOrNull == null) {
                return this;
            }
            String normalizedRootDirInArchive = Utils.normalizePath(rootDirInArchive);
            ArchiveEntry dirEntry = new ArchiveEntry(null, normalizedRootDirInArchive, linkOrNull, this.idCache);
            this.archiveDirectory(absoluteDirectory, dirEntry, strategy, chunkSize, entryVisitorOrNull);
        } else {
            this.errorStrategy.dealWithError(new ArchivingException(absoluteDirectory, new IOException("Path does not correspond to a directory.")));
        }
        return this;
    }

    public HDF5ArchiveUpdater archive(File parentDirToStrip, File path, ArchivingStrategy strategy, int chunkSize, IArchiveEntryVisitor entryVisitorOrNull) {
        boolean ok;
        File absolutePath;
        File absoluteParentDirToStrip = Utils.normalizePath(parentDirToStrip);
        String hdf5ObjectPath = HDF5ArchiveUpdater.getRelativePath(absoluteParentDirToStrip, absolutePath = Utils.normalizePath(path));
        String hdf5GroupPath = Utils.getParentPath(hdf5ObjectPath);
        boolean groupExists = hdf5GroupPath.length() == 0 ? true : this.hdf5Writer.isGroup(hdf5GroupPath);
        int crc32 = 0;
        LinkRecord linkOrNull = LinkRecord.tryCreate(absolutePath, this.errorStrategy);
        ArchiveEntry entry = new ArchiveEntry(hdf5GroupPath, hdf5ObjectPath, linkOrNull, this.idCache);
        if (linkOrNull != null && linkOrNull.isSymLink()) {
            ok = this.archiveSymLink(entry, absolutePath, entryVisitorOrNull);
        } else if (absolutePath.isDirectory()) {
            ok = this.archiveDirectory(absolutePath, entry, strategy, chunkSize, entryVisitorOrNull);
        } else if (absolutePath.isFile()) {
            DataSetInfo dataSetInfoOrNull = this.tryArchiveFile(absolutePath, entry, strategy.getStorageFeatureForPath(hdf5ObjectPath), chunkSize, entryVisitorOrNull);
            boolean bl = ok = dataSetInfoOrNull != null;
            if (dataSetInfoOrNull != null) {
                crc32 = dataSetInfoOrNull.crc32;
            }
        } else {
            ok = false;
            this.errorStrategy.dealWithError(new ArchivingException(absolutePath, new IOException("Path corresponds to neither a file nor a directory.")));
        }
        if (ok) {
            this.updateIndicesOnThePath(absoluteParentDirToStrip, absolutePath, crc32, groupExists);
        }
        return this;
    }

    private void updateIndicesOnThePath(File parentDirToStrip, File path, int crc32, boolean immediateGroupOnly) {
        String rootAbsolute = parentDirToStrip.getAbsolutePath();
        File pathProcessing = path;
        int crc32Processing = crc32;
        do {
            File dirProcessingOrNull;
            String dirAbsolute;
            String string = dirAbsolute = (dirProcessingOrNull = pathProcessing.getParentFile()) != null ? dirProcessingOrNull.getAbsolutePath() : "";
            if (dirProcessingOrNull == null || !dirAbsolute.startsWith(rootAbsolute)) break;
            String hdf5GroupPath = HDF5ArchiveUpdater.getRelativePath(rootAbsolute, dirAbsolute);
            IDirectoryIndex index = this.indexProvider.get(hdf5GroupPath, false);
            LinkRecord linkOrNull = LinkRecord.tryCreate(pathProcessing, this.errorStrategy);
            if (linkOrNull != null) {
                linkOrNull.setCrc32(crc32Processing);
                crc32Processing = 0;
                index.updateIndex(linkOrNull);
            }
            pathProcessing = dirProcessingOrNull;
        } while (!immediateGroupOnly);
    }

    private void updateIndicesOnThePath(String path, LinkRecord link, boolean immediateGroupOnly) {
        String pathProcessing = Utils.normalizePath(path);
        if ("/".equals(pathProcessing)) {
            return;
        }
        int crc32 = link.getCrc32();
        long size = link.getSize();
        long lastModified = link.getLastModified();
        short permissions = link.getPermissions();
        int uid = link.getUid();
        int gid = link.getGid();
        FileLinkType fileLinkType = link.getLinkType();
        String linkTargetOrNull = link.tryGetLinkTarget();
        do {
            String hdf5GroupPath = Utils.getParentPath(pathProcessing);
            IDirectoryIndex index = this.indexProvider.get(hdf5GroupPath, false);
            String hdf5FileName = Utils.getName(pathProcessing);
            LinkRecord linkProcessing = new LinkRecord(hdf5FileName, linkTargetOrNull, fileLinkType, size, lastModified, uid, gid, permissions, crc32);
            index.updateIndex(linkProcessing);
            fileLinkType = FileLinkType.DIRECTORY;
            crc32 = 0;
            size = -1L;
            linkTargetOrNull = null;
            pathProcessing = hdf5GroupPath;
        } while (!immediateGroupOnly && !"/".equals(pathProcessing));
    }

    private boolean archiveEmptyDirectory(String parentDirectory, LinkRecord link) {
        String hdf5GroupPath = Utils.concatLink(parentDirectory, link.getLinkName());
        try {
            this.hdf5Writer.object().createGroup(hdf5GroupPath);
            return true;
        }
        catch (HDF5Exception ex) {
            this.errorStrategy.dealWithError(new ArchivingException(hdf5GroupPath, ex));
            return false;
        }
    }

    private boolean archiveDirectory(File dir, ArchiveEntry dirEntry, ArchivingStrategy strategy, int chunkSize, IArchiveEntryVisitor entryVisitorOrNull) {
        File[] fileEntries = dir.listFiles();
        if (fileEntries == null) {
            this.errorStrategy.dealWithError(new ArchivingException(dir, new IOException("Cannot read directory")));
            return false;
        }
        String hdf5GroupPath = dirEntry.getPath();
        if (!"/".equals(hdf5GroupPath)) {
            try {
                if (this.hdf5Writer.file().getFileFormat() != IHDF5WriterConfigurator.FileFormat.STRICTLY_1_8 && fileEntries.length > 100) {
                    int totalLength = HDF5ArchiveUpdater.computeSizeHint(fileEntries);
                    this.hdf5Writer.object().createGroup(hdf5GroupPath, totalLength * 5);
                } else {
                    this.hdf5Writer.object().createGroup(hdf5GroupPath);
                }
            }
            catch (HDF5Exception ex) {
                this.errorStrategy.dealWithError(new ArchivingException(hdf5GroupPath, ex));
            }
        }
        List<LinkRecord> linkEntries = DirectoryIndex.convertFilesToLinks(fileEntries, this.errorStrategy);
        if (entryVisitorOrNull != null) {
            entryVisitorOrNull.visit(dirEntry);
        }
        Iterator<LinkRecord> linkIt = linkEntries.iterator();
        int i = 0;
        while (i < fileEntries.length) {
            File file = fileEntries[i];
            LinkRecord link = linkIt.next();
            if (link == null) {
                linkIt.remove();
            } else {
                String absoluteEntry = file.getAbsolutePath();
                ArchiveEntry entry = new ArchiveEntry(hdf5GroupPath, Utils.concatLink(hdf5GroupPath, link.getLinkName()), link, this.idCache);
                if (entry.isDirectory()) {
                    if (strategy.doExclude(absoluteEntry, true)) {
                        linkIt.remove();
                    } else {
                        boolean ok = this.archiveDirectory(file, entry, strategy, chunkSize, entryVisitorOrNull);
                        if (!ok) {
                            linkIt.remove();
                        }
                    }
                } else if (strategy.doExclude(absoluteEntry, false)) {
                    linkIt.remove();
                } else if (entry.isSymLink()) {
                    boolean ok = this.archiveSymLink(entry, file, entryVisitorOrNull);
                    if (!ok) {
                        linkIt.remove();
                    }
                } else if (entry.isRegularFile()) {
                    DataSetInfo dataSetInfoOrNull = this.tryArchiveFile(file, entry, strategy.getStorageFeatureForPath(entry.getPath()), chunkSize, entryVisitorOrNull);
                    if (dataSetInfoOrNull == null) {
                        linkIt.remove();
                    } else {
                        link.setSize(dataSetInfoOrNull.size);
                        link.setCrc32(dataSetInfoOrNull.crc32);
                    }
                } else {
                    this.errorStrategy.dealWithError(new ArchivingException(file, new IOException("Path corresponds to neither a file nor a directory.")));
                }
            }
            ++i;
        }
        boolean verbose = entryVisitorOrNull != null;
        IDirectoryIndex index = this.indexProvider.get(hdf5GroupPath, verbose);
        index.updateIndex(linkEntries);
        return true;
    }

    private boolean archiveSymLink(ArchiveEntry entry) {
        if (!entry.hasLinkTarget()) {
            this.errorStrategy.dealWithError(new ArchivingException(entry.getName(), new IOException("Link target not given for symbolic link.")));
            return false;
        }
        return this.archiveSymLink(entry, null);
    }

    private boolean archiveSymLink(ArchiveEntry entry, File file, IArchiveEntryVisitor entryVisitorOrNull) {
        if (!entry.hasLinkTarget()) {
            this.errorStrategy.dealWithError(new ArchivingException(file, new IOException("Cannot read link target of symbolic link.")));
            return false;
        }
        return this.archiveSymLink(entry, entryVisitorOrNull);
    }

    private boolean archiveSymLink(ArchiveEntry entry, IArchiveEntryVisitor entryVisitorOrNull) {
        try {
            this.hdf5Writer.object().createSoftLink(entry.getLinkTarget(), entry.getPath());
            if (entryVisitorOrNull != null) {
                entryVisitorOrNull.visit(entry);
            }
        }
        catch (HDF5Exception ex) {
            this.errorStrategy.dealWithError(new ArchivingException(entry.getPath(), ex));
            return false;
        }
        return true;
    }

    private static int computeSizeHint(File[] entries) {
        int totalLength = 0;
        File[] fileArray = entries;
        int n = entries.length;
        int n2 = 0;
        while (n2 < n) {
            File entry = fileArray[n2];
            totalLength += entry.getName().length();
            ++n2;
        }
        return totalLength;
    }

    private DataSetInfo tryArchiveFile(File file, ArchiveEntry entry, HDF5GenericStorageFeatures features, int chunkSize, IArchiveEntryVisitor entryVisitorOrNull) throws ArchivingException {
        DataSetInfo info = null;
        try {
            info = this.copyToHDF5(file, entry.getPath(), features, chunkSize);
            entry.setDataSetInfo(info);
            if (entryVisitorOrNull != null) {
                entryVisitorOrNull.visit(entry);
            }
        }
        catch (IOException ex) {
            this.errorStrategy.dealWithError(new ArchivingException(file, ex));
        }
        catch (HDF5Exception ex) {
            this.errorStrategy.dealWithError(new ArchivingException(entry.getPath(), ex));
        }
        return info;
    }

    static String getRelativePath(File root, File filePath) {
        return HDF5ArchiveUpdater.getRelativePath(root.getAbsolutePath(), filePath.getAbsolutePath());
    }

    static String getRelativePath(String parentDirToBeStripped, String filePath) {
        if (!filePath.startsWith(parentDirToBeStripped) && !parentDirToBeStripped.startsWith(filePath)) {
            throw new IOExceptionUnchecked("Path '" + filePath + "' does not start with '" + parentDirToBeStripped + "'.");
        }
        String path = parentDirToBeStripped.length() >= filePath.length() ? "/" : filePath.substring(parentDirToBeStripped.length());
        return FilenameUtils.separatorsToUnix(path);
    }

    private DataSetInfo copyToHDF5(File source, String objectPath, HDF5GenericStorageFeatures compression, int chunkSize) throws IOException {
        FileInputStream input = FileUtils.openInputStream(source);
        try {
            DataSetInfo dataSetInfo = this.copyToHDF5(input, objectPath, compression, chunkSize);
            return dataSetInfo;
        }
        finally {
            IOUtils.closeQuietly(input);
        }
    }

    private int getEffectiveChunkSize(int chunkSize) {
        return chunkSize <= 0 || chunkSize > this.buffer.length ? this.buffer.length : chunkSize;
    }

    private DataSetInfo copyToHDF5(InputStream input, String objectPath, HDF5GenericStorageFeatures compression, int chunkSize) throws IOException {
        int effectiveBufferLength = this.getEffectiveChunkSize(chunkSize);
        CRC32 crc32 = new CRC32();
        HDF5GenericStorageFeatures features = compression;
        int n = this.fillBuffer(input, effectiveBufferLength);
        if (n < effectiveBufferLength) {
            if (n <= 4096 || !features.isDeflating()) {
                features = HDF5GenericStorageFeatures.GENERIC_CONTIGUOUS;
            }
            HDF5OpaqueType type = this.hdf5Writer.opaque().createArray(objectPath, OPAQUE_TAG_FILE, n, features);
            this.hdf5Writer.opaque().writeArrayBlockWithOffset(objectPath, type, this.buffer, n, 0L);
            crc32.update(this.buffer, 0, n);
            return new DataSetInfo(n, (int)crc32.getValue());
        }
        HDF5OpaqueType type = this.hdf5Writer.opaque().createArray(objectPath, OPAQUE_TAG_FILE, 0L, effectiveBufferLength, compression);
        long count = 0L;
        while (n > 0) {
            this.hdf5Writer.opaque().writeArrayBlockWithOffset(objectPath, type, this.buffer, n, count);
            count += (long)n;
            crc32.update(this.buffer, 0, n);
            n = this.fillBuffer(input, effectiveBufferLength);
        }
        return new DataSetInfo(count, (int)crc32.getValue());
    }

    private int fillBuffer(InputStream input, int bufferLength) throws IOException {
        int ofs = 0;
        int len = bufferLength;
        int count = 0;
        int n = 0;
        while (len > 0 && -1 != (n = input.read(this.buffer, ofs, len))) {
            ofs += n;
            len -= n;
            count += n;
        }
        return count;
    }

    static class DataSetInfo {
        final long size;
        final int crc32;

        DataSetInfo(long size, int crc32) {
            this.size = size;
            this.crc32 = crc32;
        }
    }

    private final class H5ARIOutputStream
    implements IOutputStream,
    Flushable {
        private final IOutputStream delegate;
        private final String directory;
        private final String path;
        private final LinkRecord link;
        private final CRC32 crc32 = new CRC32();
        private long size = 0L;

        H5ARIOutputStream(String normalizedDirectory, LinkRecord link, int chunkSize, boolean compress) {
            this.directory = normalizedDirectory;
            this.path = Utils.concatLink(this.directory, link.getLinkName());
            this.link = link;
            HDF5GenericStorageFeatures creationStorageFeature = compress ? HDF5GenericStorageFeatures.GENERIC_DEFLATE : HDF5GenericStorageFeatures.GENERIC_NO_COMPRESSION;
            this.delegate = HDF5IOAdapterFactory.asIOutputStream(HDF5ArchiveUpdater.this.hdf5Writer, this.path, creationStorageFeature, HDF5ArchiveUpdater.this.getEffectiveChunkSize(chunkSize), HDF5ArchiveUpdater.OPAQUE_TAG_FILE);
            HDF5ArchiveUpdater.this.indexProvider.get(normalizedDirectory, false).addFlushable(this);
        }

        @Override
        public void write(int b) throws IOExceptionUnchecked {
            this.crc32.update(b);
            ++this.size;
            this.delegate.write(b);
        }

        @Override
        public void write(byte[] b) throws IOExceptionUnchecked {
            this.crc32.update(b);
            this.size += (long)b.length;
            this.delegate.write(b);
        }

        @Override
        public void write(byte[] b, int off, int len) throws IOExceptionUnchecked {
            this.crc32.update(b, off, len);
            this.size += (long)len;
            this.delegate.write(b, off, len);
        }

        @Override
        public void flush() throws IOExceptionUnchecked {
            this.link.setCrc32((int)this.crc32.getValue());
            this.link.setSize(this.size);
            boolean updateImmediateGroupOnly = HDF5ArchiveUpdater.this.hdf5Writer.isGroup(this.directory);
            HDF5ArchiveUpdater.this.updateIndicesOnThePath(this.path, this.link, updateImmediateGroupOnly);
            this.delegate.flush();
        }

        @Override
        public void synchronize() throws IOExceptionUnchecked {
            this.delegate.synchronize();
        }

        @Override
        public void close() throws IOExceptionUnchecked {
            this.flush();
            this.delegate.close();
            HDF5ArchiveUpdater.this.indexProvider.get(this.path, false).removeFlushable(this);
        }
    }
}

