/*
 * Decompiled with CFR 0.152.
 */
package ghidra.file.formats.iso9660;

import ghidra.app.cmd.comments.SetCommentCmd;
import ghidra.app.cmd.data.CreateDataCmd;
import ghidra.app.cmd.data.CreateStringCmd;
import ghidra.app.services.AbstractAnalyzer;
import ghidra.app.services.AnalyzerType;
import ghidra.app.util.bin.BinaryReader;
import ghidra.app.util.bin.ByteProvider;
import ghidra.app.util.bin.MemoryByteProvider;
import ghidra.app.util.importer.MessageLog;
import ghidra.file.formats.iso9660.ISO9660BaseVolume;
import ghidra.file.formats.iso9660.ISO9660Constants;
import ghidra.file.formats.iso9660.ISO9660Directory;
import ghidra.file.formats.iso9660.ISO9660Header;
import ghidra.file.formats.iso9660.ISO9660PathTable;
import ghidra.file.formats.iso9660.ISO9660VolumeDescriptor;
import ghidra.framework.model.DomainObject;
import ghidra.framework.options.Options;
import ghidra.program.model.address.Address;
import ghidra.program.model.address.AddressSetView;
import ghidra.program.model.address.AddressSpace;
import ghidra.program.model.data.AlignmentDataType;
import ghidra.program.model.data.DataType;
import ghidra.program.model.data.StringDataType;
import ghidra.program.model.listing.Data;
import ghidra.program.model.listing.Group;
import ghidra.program.model.listing.Program;
import ghidra.program.model.listing.ProgramFragment;
import ghidra.program.model.listing.ProgramModule;
import ghidra.program.model.mem.MemoryBlock;
import ghidra.util.Msg;
import ghidra.util.exception.CancelledException;
import ghidra.util.exception.DuplicateNameException;
import ghidra.util.task.TaskMonitor;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Set;

public class ISO9660Analyzer
extends AbstractAnalyzer {
    public ISO9660Analyzer() {
        super("ISO9660 File Format Annotation", "Annotates an ISO9660 File Format", AnalyzerType.BYTE_ANALYZER);
        super.setPrototype();
    }

    public boolean canAnalyze(Program program) {
        Offset result = this.checkSignatures(program);
        return !result.equals((Object)Offset.NotFound);
    }

    private Offset checkSignatures(Program program) {
        int magicLen = ISO9660Constants.MAGIC_BYTES.length;
        byte[] signatureArray = new byte[magicLen];
        try {
            Options options = program.getOptions("Program Information");
            String format = options.getString("Executable Format", null);
            if (!"Raw Binary".equals(format)) {
                return Offset.NotFound;
            }
            MemoryBlock[] blocks = program.getMemory().getBlocks();
            if (blocks.length != 1) {
                return Offset.NotFound;
            }
            AddressSpace addressSpace = program.getAddressFactory().getDefaultAddressSpace();
            if (!blocks[0].getStart().getAddressSpace().equals(addressSpace)) {
                return Offset.NotFound;
            }
            long blockSize = blocks[0].getSize();
            if (blocks[0].getStart().getOffset() != 0L) {
                return Offset.NotFound;
            }
            if (!blocks[0].isInitialized()) {
                return Offset.NotFound;
            }
            MemoryByteProvider provider = new MemoryByteProvider(program.getMemory(), addressSpace);
            BinaryReader reader = new BinaryReader((ByteProvider)provider, true);
            if (blockSize < 34816L) {
                return Offset.NotFound;
            }
            reader.setPointerIndex(32769);
            signatureArray = reader.readNextByteArray(magicLen);
            if (Arrays.equals(signatureArray, ISO9660Constants.MAGIC_BYTES)) {
                return Offset.Offset1;
            }
            if (blockSize < 36864L) {
                return Offset.NotFound;
            }
            reader.setPointerIndex(34817);
            signatureArray = reader.readNextByteArray(magicLen);
            if (Arrays.equals(signatureArray, ISO9660Constants.MAGIC_BYTES)) {
                return Offset.Offset2;
            }
            if (blockSize < 38912L) {
                return Offset.NotFound;
            }
            reader.setPointerIndex(36865);
            signatureArray = reader.readNextByteArray(magicLen);
            if (Arrays.equals(signatureArray, ISO9660Constants.MAGIC_BYTES)) {
                return Offset.Offset3;
            }
        }
        catch (Exception e) {
            Msg.error((Object)((Object)this), (Object)"Error when checking for ISO9660 file signatures", (Throwable)e);
        }
        return Offset.NotFound;
    }

    public boolean added(Program program, AddressSetView set, TaskMonitor monitor, MessageLog log) throws CancelledException {
        MemoryByteProvider provider = new MemoryByteProvider(program.getMemory(), program.getAddressFactory().getDefaultAddressSpace());
        BinaryReader reader = new BinaryReader((ByteProvider)provider, true);
        try {
            Offset signatureOffset = this.checkSignatures(program);
            this.setPointerOffset(signatureOffset, reader);
            monitor.setMessage("Processing ISO9660 Header");
            ISO9660Header isoHeader = new ISO9660Header(reader);
            ArrayList<ISO9660BaseVolume> volumes = isoHeader.getVolumeDescriptorSet();
            this.setPlateComment(program, this.toAddress(program, 0L), isoHeader.toString());
            ProgramModule descriptorModule = program.getListing().getDefaultRootModule().createModule("Volume Descriptors");
            this.setDescriptorData(program, volumes, descriptorModule);
            this.processPathTables(isoHeader, reader, program);
            int offset = this.getOffsetValue(signatureOffset);
            program.getListing().createData(this.toAddress(program, 0L), (DataType)new AlignmentDataType(), offset - 1);
            ISO9660VolumeDescriptor pvd = isoHeader.getPrimaryVolumeDescriptor();
            ISO9660Directory entryDir = isoHeader.getPrimaryDirectory();
            short logicalBlockSize = pvd.getLogicalBlockSizeLE();
            List<ISO9660Directory> dirList = this.createDirectoryList(reader, entryDir, logicalBlockSize);
            this.createDirectories(reader, program, dirList, logicalBlockSize);
        }
        catch (Exception e) {
            log.appendException((Throwable)e);
            return false;
        }
        return true;
    }

    private void setPointerOffset(Offset offset, BinaryReader reader) {
        if (offset.equals((Object)Offset.Offset1)) {
            reader.setPointerIndex(32768);
        } else if (offset.equals((Object)Offset.Offset2)) {
            reader.setPointerIndex(34816);
        } else {
            reader.setPointerIndex(36864);
        }
    }

    private int getOffsetValue(Offset offsetEnum) {
        if (offsetEnum.equals((Object)Offset.Offset1)) {
            return 32769;
        }
        if (offsetEnum.equals((Object)Offset.Offset2)) {
            return 34817;
        }
        return 36865;
    }

    private void setDescriptorData(Program program, List<ISO9660BaseVolume> volumes, ProgramModule descriptorModule) throws DuplicateNameException, IOException, Exception {
        for (ISO9660BaseVolume descriptor : volumes) {
            long volumeIndex = descriptor.getVolumeIndex();
            DataType descriptorDataType = descriptor.toDataType();
            Address volumeAddress = this.toAddress(program, volumeIndex);
            Data descriptorData = this.createData(program, this.toAddress(program, volumeIndex), descriptorDataType);
            this.setPlateComment(program, volumeAddress, descriptor.toString());
            this.createFragment(program, descriptorModule, descriptorDataType.getName(), descriptorData.getMinAddress(), descriptorData.getMaxAddress().next());
        }
    }

    private void processPathTables(ISO9660Header isoHeader, BinaryReader reader, Program program) throws DuplicateNameException {
        ProgramModule pathTableModule = program.getListing().getDefaultRootModule().createModule("Path Tables");
        try {
            HashMap<Integer, Short> typeLTable = isoHeader.getTypeLIndexSizeTable();
            this.createPathTableData(reader, program, pathTableModule, typeLTable, true);
            HashMap<Integer, Short> typeMTable = isoHeader.getTypeMIndexSizeTable();
            this.createPathTableData(reader, program, pathTableModule, typeMTable, false);
            HashMap<Integer, Short> supplTypeLTable = isoHeader.getSupplTypeLIndexSizeTable();
            this.createPathTableData(reader, program, pathTableModule, supplTypeLTable, true);
            HashMap<Integer, Short> supplTypeMTable = isoHeader.getSupplTypeMIndexSizeTable();
            this.createPathTableData(reader, program, pathTableModule, supplTypeMTable, false);
        }
        catch (Exception e) {
            e.printStackTrace();
        }
    }

    private List<ISO9660Directory> createDirectoryList(BinaryReader reader, ISO9660Directory parentDir, long blockSize) throws IOException {
        long dirIndex;
        ArrayList<ISO9660Directory> directoryList = new ArrayList<ISO9660Directory>();
        ISO9660Directory childDir = null;
        long endIndex = dirIndex + (long)parentDir.getDataLengthLE();
        for (dirIndex = (long)parentDir.getLocationOfExtentLE() * blockSize; dirIndex < endIndex; dirIndex += (long)childDir.getDirectoryRecordLength()) {
            reader.setPointerIndex(dirIndex);
            if (reader.peekNextByte() != 0) {
                childDir = new ISO9660Directory(reader, parentDir);
                directoryList.add(childDir);
                continue;
            }
            while (reader.peekNextByte() == 0 && reader.getPointerIndex() < endIndex) {
                reader.readNextByte();
            }
            if (reader.getPointerIndex() >= endIndex) continue;
            childDir = new ISO9660Directory(reader, parentDir);
            dirIndex = childDir.getVolumeIndex();
            directoryList.add(childDir);
        }
        return directoryList;
    }

    private void createDirectories(BinaryReader reader, Program program, List<ISO9660Directory> directoryList, long blockSize) throws DuplicateNameException, Exception {
        if (directoryList.size() > 2) {
            ISO9660Directory selfDir = null;
            ISO9660Directory parentDir = null;
            selfDir = directoryList.remove(0);
            Address volumeAddress = this.toAddress(program, selfDir.getVolumeIndex());
            this.createDataAndPlateComment(program, selfDir, volumeAddress);
            parentDir = directoryList.remove(0);
            volumeAddress = this.toAddress(program, parentDir.getVolumeIndex());
            this.createDataAndPlateComment(program, parentDir, volumeAddress);
            for (ISO9660Directory dir : directoryList) {
                if (!selfDir.isDirectoryFlagSet()) continue;
                volumeAddress = this.toAddress(program, dir.getVolumeIndex());
                this.setPlateComment(program, volumeAddress, dir.toString());
                DataType volumeDataType = dir.toDataType();
                this.createData(program, volumeAddress, volumeDataType);
                if (!dir.isDirectoryFlagSet()) continue;
                List<ISO9660Directory> dirs = this.createDirectoryList(reader, dir, blockSize);
                this.createDirectories(reader, program, dirs, blockSize);
            }
        }
    }

    private void createDataAndPlateComment(Program program, ISO9660Directory dir, Address volumeAddress) throws DuplicateNameException, IOException, Exception {
        this.setPlateComment(program, volumeAddress, dir.toString());
        this.createData(program, volumeAddress, dir.toDataType());
    }

    private void createPathTableData(BinaryReader reader, Program program, ProgramModule module, HashMap<Integer, Short> pathTableMap, boolean littleEndian) throws Exception {
        Set<Integer> pathTableIndexes = pathTableMap.keySet();
        for (int pathTableIndex : pathTableIndexes) {
            short logicalBlockSize = pathTableMap.get(pathTableIndex);
            int pathAddress = logicalBlockSize * pathTableIndex;
            reader.setPointerIndex(pathAddress);
            ISO9660PathTable pathTable = new ISO9660PathTable(reader, littleEndian);
            DataType pathTableDataType = pathTable.toDataType();
            Address volumeAddress = this.toAddress(program, pathTable.getVolumeIndex());
            this.setPlateComment(program, volumeAddress, pathTable.toString());
            Data pathTableData = this.createData(program, volumeAddress, pathTableDataType);
            this.createFragment(program, module, pathTableDataType.getName(), pathTableData.getMinAddress(), pathTableData.getMaxAddress().next());
        }
    }

    private Data createData(Program program, Address address, DataType datatype) throws Exception {
        if (datatype instanceof StringDataType) {
            CreateStringCmd cmd = new CreateStringCmd(address);
            if (!cmd.applyTo((DomainObject)program)) {
                throw new RuntimeException(cmd.getStatusMsg());
            }
        } else {
            CreateDataCmd cmd = new CreateDataCmd(address, datatype);
            if (!cmd.applyTo((DomainObject)program)) {
                throw new RuntimeException(cmd.getStatusMsg());
            }
        }
        return program.getListing().getDefinedDataAt(address);
    }

    private Address toAddress(Program program, long offset) {
        return program.getAddressFactory().getDefaultAddressSpace().getAddress(offset);
    }

    private boolean setPlateComment(Program program, Address address, String comment) {
        SetCommentCmd cmd = new SetCommentCmd(address, 3, comment);
        return cmd.applyTo((DomainObject)program);
    }

    private ProgramFragment createFragment(Program program, ProgramModule module, String fragmentName, Address start, Address end) throws Exception {
        ProgramFragment fragment = this.getFragment(module, fragmentName);
        if (fragment == null) {
            fragment = module.createFragment(fragmentName);
        }
        fragment.move(start, end.subtract(1L));
        return fragment;
    }

    private ProgramFragment getFragment(ProgramModule module, String fragmentName) {
        Group[] groups = module.getChildren();
        if (groups != null) {
            for (Group group : groups) {
                if (!group.getName().equals(fragmentName) || !(group instanceof ProgramFragment)) continue;
                return (ProgramFragment)group;
            }
        }
        return null;
    }

    private static enum Offset {
        Offset1,
        Offset2,
        Offset3,
        NotFound;

    }
}

