/*
    Written on 2004 by Bruno Vedder

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation
; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/*****************************************************************************
*
* File: main.cpp
*
* Project: Osmose emulator.
*
* Author: Vedder Bruno
* Date: 28/10/2004, 13h00
*
* URL: http://bcz.emu-france.com/
*****************************************************************************/

#include <iostream>
#include <iomanip>
#include "OsmoseCore.h"
#include "Options.h"
#include "Version.h"
#include "Definitions.h"

using namespace std;


// Functions prototypes.
void usage();
int parseCommandLine(int a, char *rv[]);
void splashText();
unsigned int timer_callback(unsigned int i, void *p);

Options              opt;        // SMS Machine related options.
EmulatorOptions      emu_opt;    // Emulator related options.
SDL_sem           *semaphore;      // Semaphore used for FPS synchronisation.

/*--------------------------------------------------------------------*/
/* This is the main method of osmose. It check args number, correct   */
/* Emulator options, and start emulation is Ok.                       */
/*--------------------------------------------------------------------*/
int main(int argc,char *argv[])
{
	int ret;
    unsigned int frame = 0;
	bool quit = false;
	SDL_Event event;
	SDL_TimerID         t_id;        // SDL Timer used for 60hz synchro.

	unsigned int start_time;
	unsigned int stop_time;
	static char fps_string[16];
	
    // Reset default options.
    emu_opt.reset();
    opt.reset();

    // Display Osmose version, author.
    splashText();

    // Check arguments number. If not enough display usage.
    if (argc < 2)
    {
        usage();
        exit(0);
    }

    // emu_opt will be corrected depending on cmd line.
    ret = parseCommandLine(argc, argv);

    /* ROM  filename ending with .gg .sms .zip has been found at index ret in argv[] */
    if (ret)
    {
        OsmoseCore core(argv[ret]);
    	semaphore = SDL_CreateSemaphore(0);
    	t_id = SDL_AddTimer(DELAY_BETWEEN_FRAME, timer_callback, NULL);
        cout << "Starting emulation." << endl;

		start_time = SDL_GetTicks();

		float on_fly_start = start_time;
		float on_fly_stop;
		SDL_WM_SetCaption(__OSMOSE_VERSION__,__OSMOSE_VERSION__);   // Window title, Iconified widows title */        		
		
    	//TW->ADDTEXT(__OSMOSE_VERSION__,120);
        while(!quit)
    	{
			if (frame % 3 == 0)
			{
				SDL_SemWait(semaphore);
			}
			/* Handle SDL Events */
			while( SDL_PollEvent( &event ) )
			{
		   		if ((event.type == SDL_KEYDOWN) && ( event.key.keysym.sym == SDLK_ESCAPE)) quit = true;
				if (event.type == SDL_QUIT) quit = true;
				if (event.type == SDL_VIDEORESIZE) core.Reshape(event.resize.w,event.resize.h);
				else core.handleSDLKeyboardEvent(event);
			}
        	core.run_frame();
        	frame++;
        	
        	if (emu_opt.display_fps == true)
        	{
	        	if ((frame & 63) == 0)
    	    	{     	
    	    		on_fly_stop = SDL_GetTicks();
    	    		float instant_fps = ( 1000.0 / (float)((on_fly_stop - on_fly_start) / 64.0));
    	    		sprintf(fps_string, "%.2f fps", instant_fps);
    	    		SDL_WM_SetCaption(fps_string, fps_string);
    	    		on_fly_start = on_fly_stop;   	    		    	    		
        		} 
        	}
    	}

		stop_time = SDL_GetTicks();

    	SDL_CloseAudio();
    	core.save_bbr();
    	//// WE MAY BE RECORDING SOUND WHEN LEAVING EMULATION. IF IT'S THE CASE, CLOSE SOUND_SHOT FILE.
    	//IF (SOUND_SHOT_TOGGLE == TRUE && EMU_OPT.SOUND == TRUE)
    	//{
        //WAVW->CLOSE();
   	 	//}
    	SDL_Quit();

	    // Now show statistics:
	    float emu_time = (float)((float)(stop_time - start_time) / 1000.0);
	    float a_fps;
	    
	    if (emu_time > 0)
	    {
	    	a_fps = frame / emu_time;
 	    }
 	    else
 	    {
 	    	a_fps = 0;
 	    }
 	    
 	    cout << "Total rendered frames    : " << frame << endl;
	    cout << "Total emulation time     : " << emu_time << " seconds." << endl;
 	    cout << "Average frame per second : " << a_fps << " frames per second." <<endl;
	}
    else
    {
    	cout << "No ROM filename has been found on command line(.gg .GG .sms .SMS .zip .ZIP extensions)." << endl;
    	usage();
    }

    return 0;
}

/*--------------------------------------------------------------------*/
/* This method displays osmose usage.				      */
/*--------------------------------------------------------------------*/
void usage()
{
    cout << "Usage is osmose filename, ex: osmose asterix.zip -fs -bilinear -nosound" << endl << endl;
    cout << "Options: "<< endl;
    cout << "    -paddle           Emulates one axis paddle (mapped on mouse)." << endl;
    cout << "    -joy              Use joystick as input device, instead of keyboard." << endl;
    cout << "    -acceleration x.x Paddle acceleration (0.1 to 5 default "<< DEFAULT_ACCELERATION << ")" << endl;
    cout << "    -fs               Run in fullscreen   (default: windowed)." << endl;
    cout << "    -bilinear         Bilinear filtering on screen (default: nearest neighbour)" << endl;
    cout << "    -nosound          Do not play sounds. (default: sound on)."<< endl;
    cout << "    -dp               Use dark palette for screen emulation (default: off)." << endl;
    cout << "    -inifile xxx      Use xxx as configuration file. " <<endl;
    cout << "    -fps              Display fps in title bar." << endl;
    cout << "    -cm               Use codemaster games mem. mapper (default: off). " << endl;
    cout << "    -km               Use korean games mem. mapper (default: off). " << endl;
	cout << "    -irqhack          Enable irq hack (specific rom option. default: off)." << endl;
    cout << "    -pal              Emulates PAL/SECAM video timing (default: NTSC)."<< endl;
    cout << "    -jap              Run as japanese machine (default: exp)." << endl;
    cout << "    -exp              Run as exported machine (already default)." << endl;
}

/*--------------------------------------------------------------------*/
/* This method parses command line, and update options structure with */
/* right options.						      */
/*--------------------------------------------------------------------*/
int parseCommandLine(int a, char *rv[])
{
	int rom_filename_index = 0;
    for (int i=0; i<a; i++)
    {
        if (strcmp(rv[i],"-fs")==0)
        {
            emu_opt.fullscreen_flag = true;
        }

        if (strcmp(rv[i],"-pal")==0)
        {
            opt.ntsc = false;
        }

        if (strcmp(rv[i],"-fps")==0)
        {
            emu_opt.display_fps = true;
        }

        if (strcmp(rv[i],"-bilinear")==0)
        {
            opt.videoFilter = BILINEAR;
        }

        if (strcmp(rv[i],"-jap")==0)
        {
            // Act as Japanese machine.
            opt.WorldVersion = false;
        }

        if (strcmp(rv[i],"-exp")==0)
        {
            // Act as Exported machine.
            opt.WorldVersion = true;
        }

        if (strcmp(rv[i],"-acceleration")==0)
        {
		    // If actual argument nbr < argument number eg is there any args ?
		    if (i < a-1)
		    {
	            opt.acceleration = atof( rv[i+1]);
				if (opt.acceleration == 0)
				{
			        cerr << "No valid acceleration parameter was given. Using default " << (float)DEFAULT_ACCELERATION << " value." << endl;
		            opt.acceleration = DEFAULT_ACCELERATION;
				}
				cout << "Paddle acceleration used : " << opt.acceleration << endl;
			}
			else
			{
			    cerr << "No acceleration parameter was given. Using default " << (float)DEFAULT_ACCELERATION << " value." << endl;
		        opt.acceleration = DEFAULT_ACCELERATION;
			}
        }


        if (strcmp(rv[i],"-nosound")==0)
        {
            emu_opt.sound = false;
        }

        if (strcmp(rv[i],"-cm")==0)
        {
            opt.mapperType = CodemasterMapper;
        }

        if (strcmp(rv[i],"-km")==0)
        {
            opt.mapperType = KoreanMapper;
        }

        if (strcmp(rv[i],"-irqhack")==0)
        {
            opt.irq_hack = true;
        }

        if (strcmp(rv[i],"-paddle")==0)
        {
            opt.inputType = PADDLE;
        }

        if (strcmp(rv[i],"-joy")==0)
        {
            opt.inputType = JOYSTICK;
        }

        if (strcmp(rv[i],"-dp")==0)
        {
            emu_opt.bright_palette = false;
        }

        // The ini file MUST be the next parameter after -inifile option.
        if (strcmp(rv[i],"-inifile")==0)
        {
		    // If actual argument nbr < argument number eg is there any args ?
		    if (i < a-1)
		    {
	            emu_opt.default_config = false;
		        strncpy(emu_opt.ini_file, rv[++i],512);
		    }
		    else
		    {
		        cerr << "No ini file was given, using default configuration." << endl;
	            emu_opt.default_config = true;
		    }
        }

        // This parameter is not an option. Is it a ROM filename ?
        char extension [][5] = {".zip",".ZIP",".sms",".SMS",".gg",".GG"};
        for (int o = 0; o < 6; o++)
        {
        	char *pattern = NULL;
        	pattern = strstr(rv[i], extension[o]);
        	if (pattern != NULL)
        	{
        	    rom_filename_index = i;
        	}
        }
    }
    return (rom_filename_index);
}

/*--------------------------------------------------------------------*/
/* This method writes Osmose splash screen, version, author and build */
/* informations.						      */
/*--------------------------------------------------------------------*/
void splashText()
{
    cout << endl <<  __OSMOSE_VERSION__ <<" build on " << __DATE__<< " at " << __TIME__<< " Written by Bruno Vedder." << endl;
}

/*--------------------------------------------------------------------*/
/* This method is used to synchronise emulator at good FPS. Each      */
/* DELAY_BETWEEN_FRAME milliseconds, the semaphore is freed. Then, the*/
/* main loop can continue it's execution. Note that DELAY_.. is 50ms. */
/* This is done to avoid problems due to timer granularity. Timer<20ms*/
/* aren't very accurate. So instead of drawing frame, waiting 16ms for*/
/* three times, we draw 3 frames, and wait 50ms. 50ms is large enough */
/* to get good synchronisation.					      */
/*--------------------------------------------------------------------*/
unsigned int timer_callback(unsigned int i, void *p)
{
    SDL_SemPost(semaphore);
    return DELAY_BETWEEN_FRAME;
}
